#!/bin/bash

# apt-get install netcat-traditional dnsutils openssl -y

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[0;33m'
NC='\033[0m'

BASE_PATH="/etc/cyolo"
IPV4_REGEX='^([0-9]{1,3}\.){3}[0-9]{1,3}$'

log()
{
   echo -e $1
}

dependencies_check()
{
  for app in nc dig openssl; do
    command -v ${app}
    if (( $? )); then
       log "${app} is not installed, please install it"
       log "Note: for Ubuntu you can use the following command\napt-get install netcat-traditional dnsutils openssl -y"
       exit 127
    fi 
  done
}

extract_upstreams()
{
  if [[ -f ${BASE_PATH}/config/docker-compose.yml ]]; then
    log "Extracting values from docker-compose.yml"
    # Extract the UPSTRAEM from docker-compose.yml
    UPSTREAM=$(sed "s/\(.*\)\(--\|#\).*/\1/g" ${BASE_PATH}/config/docker-compose.yml | grep '\- UPSTREAM='| awk -F"=" '{print $2}')
    # Extract the UPSTRAEM_SNI from docker-compose.yml
    UPSTREAM_SNI=$(sed "s/\(.*\)\(--\|#\).*/\1/g" ${BASE_PATH}/config/docker-compose.yml | grep '\- UPSTREAM_SNI='| awk -F"=" '{print $2}')
    # Extract the NATIVE_SSH_UPSTREAM_ADDR from docker-compose.yml
    NATIVE_SSH_UPSTREAM_ADDR=$(sed "s/\(.*\)\(--\|#\).*/\1/g" ${BASE_PATH}/config/docker-compose.yml | grep '\- NATIVE_SSH_UPSTREAM_ADDR='| awk -F"=" '{print $2}')
    # Extract the UPSTRAEM CLOCK from docker-compose.yml
    UPSTREAM_CLOCK=$(sed "s/\(.*\)\(--\|#\).*/\1/g" ${BASE_PATH}/config/docker-compose.yml | grep '\- UPSTREAM_CLOCK='| awk -F"=" '{print $2}')
    # Extract the UPSTRAEM_SNI from docker-compose.yml
    UPSTREAM_CLOCK_SNI=$(sed "s/\(.*\)\(--\|#\).*/\1/g" ${BASE_PATH}/config/docker-compose.yml | grep '\- UPSTREAM_CLOCK_SNI='| awk -F"=" '{print $2}')
    # Extract the USE_EDGE_PROXY from docker-compose.yml
    USE_EDGE_PROXY=$(sed "s/\(.*\)\(--\|#\).*/\1/g" ${BASE_PATH}/config/docker-compose.yml | grep '\- USE_EDGE_PROXY='| awk -F"=" '{print $2}')
  fi
  if [[ -z "${UPSTREAM}" ]]; then
    UPSTREAM="tcp.cyolo.io:443"
  fi
  if [[ -z "${UPSTREAM_SNI}" ]]; then
    UPSTREAM_SNI="tcp.cyolo.io"
  fi
  if [[ -z "${NATIVE_SSH_UPSTREAM_ADDR}" ]]; then
    NATIVE_SSH_UPSTREAM_ADDR="ssh.cyolo.io:22"
  fi
  if [[ -z "${UPSTREAM_CLOCK}" ]]; then
    UPSTREAM_CLOCK="clock.cyolo.io:443"
  fi
  if [[ -z "${UPSTREAM_CLOCK_SNI}" ]]; then
    UPSTREAM_CLOCK_SNI="clock.cyolo.io"
  fi
  if [[ -z "${USE_EDGE_PROXY}" ]]; then
    USE_EDGE_PROXY=false
  fi
  # Extract the base domain (everything after the first dot)
  BASE_DOMAIN=$(echo "${UPSTREAM_SNI}" | sed -E 's/^[^.]+\.//')
  # Construct the new domain
  AFFINITY_URL="affinity.${BASE_DOMAIN}"
  # Extract the Upstream Host/IP
  UPSTREAM_HOST=$(echo ${UPSTREAM} | awk -F":" '{print $1}')
  if [[ $UPSTREAM_HOST =~ $IPV4_REGEX ]]; then
    UPSTREAM_IP=${UPSTREAM_HOST}
  else
    UPSTREAM_IP=$(dig ${UPSTREAM_HOST} +short +answer| grep '^[.0-9]*$') 
  fi
  # Extract the Upstream Port
  UPSTREAM_PORT=$(echo ${UPSTREAM} | awk -F":" '{print $2}')
  # Extract the SSH Host/IP
  SSH_HOST=$(echo ${NATIVE_SSH_UPSTREAM_ADDR} | awk -F":" '{print $1}')
  
  log "UPSTREAM is ${UPSTREAM}"
  log "UPSTREAM_SNI is ${UPSTREAM_SNI}"
  log "NATIVE_SSH_UPSTREAM_ADDR is ${NATIVE_SSH_UPSTREAM_ADDR}"
  log "AFFINITY_URL is ${AFFINITY_URL}"
  log "UPSTREAM_CLOCK is ${UPSTREAM_CLOCK}"
  log "UPSTREAM_CLOCK_SNI is ${UPSTREAM_CLOCK_SNI}"
}

check_upstream()
{
  url=$1
  port=$2
  sni=$3
  typeset -i TIMEOUT=10
  typeset -i STATUS=0

  log "Checking the ${url} DNS Resolving"
  if [[ ${url} =~ $IPV4_REGEX ]]; then
    log "The UPSTREAM provided, ${url}, is IPV4. Skipping DNS check"
    res=${url}
  else
    res=$(dig ${url} +short +answer| grep '^[.0-9]*$')
  fi
  if (( $? )); then
     log "Error> executing dig ${RED}failed${NC}"
     return
  fi
  for record in ${res}; do
    log "\t${record}"
  done
  log "Validating connection to ${url}:${port} - timeout is set for ${TIMEOUT} seconds"
  # First checking using NC, once NC is working will test using openssl
  # there is a need to add different types of validations for the openssl
  for record in ${res}; do
    nc -z -w ${TIMEOUT} ${record} ${port}
    STATUS=$?
    if (( ${STATUS} )); then
       log "\tNC testing ${record} on port ${port} - ${RED}failed${NC} (${STATUS})"
    else
       STATUS=$(echo ${openssl_res} | grep -qc CONNECTED)
       openssl s_client -connect ${record}:${port} -state -servername ${sni}  2>/dev/null | grep -q CONNECTED
       if (( ${STATUS} )); then
          log "\topenssl ${record} on port ${port} - ${RED}failed${NC} (${STATUS})"
       else
          log "\topenssl ${record} on port ${port} - ${GREEN}success${NC}"
       fi
    fi
  done
  if [ ${USE_EDGE_PROXY} == false ]; then
    log "Checking Access to Cloud Deployment (system.cyolo.io) - direct"
    res=$(curl -L -s -o /dev/null -w "CODE=%{http_code};DNS=%{time_namelookup};TOTAL=%{time_total}" https://login.system.cyolo.io)
  else
    log "Checking Access to Cloud Deployment (system.cyolo.io) via Private gateway"
    res=$(curl -L -s -o /dev/null -w "CODE=%{http_code};DNS=%{time_namelookup};TOTAL=%{time_total}" --resolve login.system.cyolo.io:443:${UPSTREAM_IP} https://login.system.cyolo.io )
  fi
  if (( $? )); then
     log "Error> Connection to Cloud (system.cyolo.io) ${RED}failed${NC}"
  else
     eval ${res}
     log "\tHTTP Status Code: ${GREEN}${CODE}${NC}"
     log "\tDNS Resolving: ${GREEN}${DNS}s${NC}"
     log "\tTotal Time: ${GREEN}${TOTAL}s${NC}"
  fi
}

check_affinity()
{
  typeset -i TIMEOUT=10
  typeset -i STATUS=0
  log "Affinity ${AFFINITY_URL} results:"
  if [[ "${AFFINITY_URL}" == "affinity.cyolo.io" ]]; then
    res=$(curl -L -s https://${AFFINITY_URL} | sed -e 's/"//g'| sed -e 's/:/ /')
  else
    res=$(curl -L -s --connect-timeout ${TIMEOUT} --resolve ${AFFINITY_URL}:443:${UPSTREAM_IP}  https://${AFFINITY_URL} | sed -e 's/"//g'| sed -e 's/:/ /' )
  fi
  if (( $? )); then
     log "Error> getting affinity ${RED}failed${NC}"
     return
  fi
  log "\t${res}"
  log "Validating connection to the Affinity - timeout is set for ${TIMEOUT} seconds"
  nc -z -w ${TIMEOUT} ${res}
  STATUS=$?
  if (( ${STATUS} )); then
     log "\taccessing ${res} - ${RED}failed${NC} (${STATUS})"
  else
     log "\taccessing ${res} - ${GREEN}success${NC}"
  fi
  log "Latency check for Affinity"
  if [[ "${AFFINITY_URL}" == "affinity.cyolo.io" ]]; then
    res=$(curl -L -s -o /dev/null -w "CODE=%{http_code};DNS=%{time_namelookup};TOTAL=%{time_total}" https://${AFFINITY_URL})
  else
    res=$(curl -L -s -o /dev/null -w "CODE=%{http_code};DNS=%{time_namelookup};TOTAL=%{time_total}" --resolve ${AFFINITY_URL}:443:${UPSTREAM_IP}  https://${AFFINITY_URL} )
  fi
  if (( $? )); then
     log "Error> affinity latency check has ${RED}failed${NC}"
  else
     eval ${res}
     log "\tHTTP Status Code: ${GREEN}${CODE}${NC}"
     log "\tDNS Resolving: ${GREEN}${DNS}s${NC}"
     log "\tTotal Time: ${GREEN}${TOTAL}s${NC}"
  fi
}

check_docker()
{
  log "Get Docker Version"
  res=$(command -v docker > /dev/null && docker --version)
  if (( $? )); then
     log "Error> getting docker version failed"
     return
  fi
  log "\t${res}"
  log "Get Docker Compose Version"
  res=$(command -v docker-compose > /dev/null && docker-compose --version)
  if (( $? )); then
     log "Error> getting docker-compose version failed"
     return
  fi
  log "\t${res}"
}

check_url()
{
  url=$1
  http_status=$2
  http_method=$3
  res=$(curl -X ${http_method} -L -s -o /dev/null -w "%{http_code}" https://${url})
  if (( $? )); then
     log "Error> accessing ${url} ${RED}failed${NC}"
     return
  fi
  if [[ "${res}" == "${http_status}" ]]; then
     log "Accessing ${url} results - ${GREEN}success${NC}"
  else
     log "Accessing ${url} results - ${RED}failed${NC} (${res})"
  fi
}

check_no_tls_inspection()
{
  res=$(echo | openssl s_client -showcerts -servername registry.docker.io -connect registry-1.docker.io:443 2>/dev/null | openssl x509 -inform pem -noout -issuer)
  if (( $? )); then
     log "Error> openssl test ${RED}failed${NC}"
     return
  fi
  if [[ ${res} == *Amazon* ]]; then
     log "openssl test results - ${GREEN}success${NC}"
  else
     log "openssl test results - ${RED}failed${NC} (${res})"
  fi
}

check_clock () 
{
	SysTimeSec=`date +%s`
        SysTimeDate=`date +%Y-%m-%dT%H:%M:%S`
	if [ ${USE_EDGE_PROXY} == false ]; then
                log "Checking Access to Cyolo Clock - direct"
		CyoloClockSec=$(curl -s --connect-timeout 3 https://${UPSTREAM_CLOCK})
	else
                log "Checking Access to Cyolo Clock via Private gateway"
		CyoloClockSec=$(curl -s --connect-timeout 3 --resolve ${UPSTREAM_CLOCK}:443:${UPSTREAM_IP} https://${UPSTREAM_CLOCK})
	fi
        if (( $? )); then
           log "Error> a call to ${UPSTREAM_CLOCK} ${RED}failed${NC}"
           return
        fi
	if [ "$CyoloClockSec" != "" ]; then
		echo "IDAC  Clock: $SysTimeDate"
		CyoloClockDate=`date -d @$CyoloClockSec +%Y-%m-%dT%H:%M:%S`
		echo "Cyolo Clock: $CyoloClockDate"
		SecDiff=$(($SysTimeSec-$CyoloClockSec))
		SecsDiff=`echo ${SecDiff#-}`
		if [ "$SecsDiff" -gt "20" ]; then
			echo "Warning!!! Time Diff Between Cyolo Cloud/iDAC is Bigger than $SecsDiff [Seconds]"
		fi
	fi
}

check_speed()
{
  log "Check download speed"
  curl https://deploy.cyolo.io/10m.txt -o /dev/null
}

log "${YELLOW}Cyolo Check Version 2.0 ${NC}"
log "\n======= ${YELLOW} Checking for Dependencies ${NC}"
dependencies_check

log "\n======= ${YELLOW} Upstream config ${NC}"
extract_upstreams

log "\n======= ${YELLOW} Cyolo Cloud ${NC}"
check_upstream ${UPSTREAM_HOST} ${UPSTREAM_PORT} ${UPSTREAM_SNI}
check_upstream ${SSH_HOST} 443 "ssh.cyolo.io"
check_affinity
echo ""

log "\n======= ${YELLOW} Cyolo Services ${NC}"
check_upstream "metrics.services.cyolo.io" 443
check_url "services.cyolo.io" 403 GET
check_url "registry.cyolo.io/v2/cyolosec/idac/tags/list" 200 GET
echo ""

log "\n======= ${YELLOW} Speed Check ${NC}"
check_speed
echo ""

log "\n======= ${YELLOW} Docker & docker-compose ${NC}"
check_docker
echo ""

log "\n======= ${YELLOW} TLS inspection test ${NC}"
check_no_tls_inspection

log "\n======= ${YELLOW} Check Cyolo Clock ${NC}"
check_clock
